using System;
using System.Collections.Generic;
using System.Drawing;
using System.IO;

using VirtualLab.Programming;
using VirtualLab.Design;
using VirtualLabAPI.Core.BasicFunctions;
using VirtualLabAPI.Core.Common;
using VirtualLabAPI.Core.DataVisualization;
using VirtualLabAPI.Core.FieldRepresentations;
using VirtualLabAPI.Core.Functions;
using VirtualLabAPI.Core.GeometryDescription;
using VirtualLabAPI.Core.LightPath;
using VirtualLabAPI.Core.Materials;
using VirtualLabAPI.Core.Modules;
using VirtualLabAPI.Core.Numerics;
using VirtualLabAPI.Core.Numerics.Region2D;
using VirtualLabAPI.Core.OpticalSystems;
using VirtualLabAPI.Core.Propagation;
using VirtualLabAPI.Core.SupportFunctions;

namespace OwnCode {
    public class VLModule : IVLModule {
        public void Run() {
            
            // define unwrap threshold
            double TwoPiThreshold = 0.95;
            // get input field
            DataArray1D inputComplexFunction = (DataArray1D) VL_GUI.SelectOpenDataArray1D();
            // consistency check
            if(inputComplexFunction.Data.Length != 1)
            {
                Globals.DataDisplay.LogError("Input DataArray contains more than one layer data.");
            }
            else
            {
                // extract actual data from input data array
                ComplexField inputData = inputComplexFunction.Data[0];
                int SamplingPoints = inputData.SamplingPoints.X;
                // initialize output 
                ComplexField outputData = new ComplexField(new Vector(1, SamplingPoints),
                    false, // extracted phase is real-valued
                    PrecisionMode.Double);
                // first data
                outputData[0, 0] = inputData[0, 0].Arg(); // input is complex
                // loop
                int PhaseJumpCount = 0;
                for(int i = 1; i < SamplingPoints; i++)
                {
                    // get phase values
                    double previousPhaseValue = outputData[0, i - 1].Re; // output is real-valued
                    double currentPhaseValue = inputData[i, 0].Arg() + 2.0 * Math.PI * PhaseJumpCount; // assume phase increases
                    // calculate difference between current and previous
                    double phaseDifference = Math.Abs(currentPhaseValue - previousPhaseValue);
                    // compare with threshold value
                    if(phaseDifference > TwoPiThreshold * 2.0 * Math.PI)
                    {
                        PhaseJumpCount++;
                        currentPhaseValue += 2.0 * Math.PI;
                    }
                    outputData[0, i] = currentPhaseValue;
                }
                // generate output data array
                DataArray1D outputDataArray = new DataArray1D(new ComplexField1DArray(new ComplexField[1] { outputData }),
                    new PhysicalProperty[1] { PhysicalProperty.AngleRad },
                    new string[1] { "Unwrapped Phase" },
                    inputComplexFunction.SamplingDistance,
                    inputComplexFunction.CoordinateOfFirstDataPoint,
                    inputComplexFunction.PhysicalPropertyOfCoordinates,
                    inputComplexFunction.CommentOfCoordinates);
                // display
                Globals.DataDisplay.ShowDocument(outputDataArray);
            }
            
            
        }
    }
}